/**
 * This file is part of WorldFileTool.
 * 
 * JMapDesk is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * JMapDesk is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with JMapDesk.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package ch.hsr.worldfiletool;

import java.awt.Cursor;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Rectangle;
import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.InputEvent;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.awt.event.WindowEvent;
import java.awt.event.WindowListener;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.util.Arrays;
import java.util.Properties;
import java.util.regex.Pattern;

import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JTextField;
import javax.swing.KeyStroke;
import javax.swing.filechooser.FileFilter;
import javax.swing.filechooser.FileNameExtensionFilter;

/**
 * The Class GUI_Swing.
 */
public class GUI_Swing extends JFrame {
	
	private static final long serialVersionUID = 1L;
	JPanel panel;
	static JTextField name;
	static JTextField image;
	static JTextField north;
	static JTextField west;
	static JTextField south;
	static JTextField east;
	static JTextField floor;
	static JComboBox maptype;
	static JTextField priority;
	WorldFileTool wft;
	private JLabel icon;
	private File path = new File(System.getProperty("user.dir"));
	private File file;
	private Properties properties = new Properties();
	private static final String lastpath = "lastpath";
	
	/**
	 * Instantiates a new GUI_Swing.
	 */
	public GUI_Swing() {
		wft = new WorldFileTool();
	}
	
	/**
	 * Create GUI.
	 */
	void create_gui(String version) {
		load();
		setTitle("WorldFileTool " + version);
		setDefaultCloseOperation(JFrame.DO_NOTHING_ON_CLOSE);
		setResizable(false);
		setSize(285, 430);
	    setIconImage(new ImageIcon(getClass().getResource("/ch/hsr/worldfiletool/images/icon.jpg")).getImage());
		Dimension dim = Toolkit.getDefaultToolkit().getScreenSize();
		setLocation((dim.width - 250)/2, (dim.height - 600)/2);
		
		panel = new JPanel();
		panel.setLayout(null);
		
		// import kml
	    JButton button_openkml = create_button("Import File...", new Rectangle(15, 310, 120, 25), KeyEvent.VK_I);
	    button_openkml.setMnemonic(KeyEvent.VK_I);
		button_openkml.addActionListener(new ActionListener() {
			@Override
			public void actionPerformed(ActionEvent e) {
				import_file();
			}
		});
		// export kml
		JButton button_savekml = create_button("Export as KML...", new Rectangle(145, 310, 120, 25), KeyEvent.VK_K);
		button_savekml.addActionListener(new ActionListener() {
			@Override
			public void actionPerformed(ActionEvent e) {
				export_kml();
			}
		});
		// export worldfile
		JButton button_openwf = create_button("Export as World File", new Rectangle(145, 340, 120, 25), KeyEvent.VK_W);
		button_openwf.setFont(new Font("Arial", Font.PLAIN, 9));
		button_openwf.addActionListener(new ActionListener() {
			@Override
			public void actionPerformed(ActionEvent e) {
				export_worldfile();
			}
		});
		// exit
		JButton button_exit = create_button("Exit", new Rectangle(15, 340, 120, 25), KeyEvent.VK_E);
		button_exit.addActionListener(new ActionListener() {
			@Override
			public void actionPerformed(ActionEvent e) {
				save_and_exit();
			}
		});
		
		JMenuBar menuBar = menu_bar();
		setJMenuBar(menuBar);
		
		// Icon next to image
		icon = new JLabel();
		icon.setIcon(new ImageIcon(getClass().getResource("/ch/hsr/worldfiletool/images/cross.png")));
		icon.setBounds(245, 65, 20, 20);
		// Some Listener
		KeyListener numListener = new KeyListener() {
			@Override
			public void keyPressed(KeyEvent e) {}
			@Override
			public void keyReleased(KeyEvent e) {}
			@Override
			public void keyTyped(KeyEvent e) {
				int k = e.getKeyChar();
				/*
				 * 48-57: 0-9
				 * 45: minus
				 * 46: dot
				 */
				if (!(k > 44 && k < 58)) {
					e.setKeyChar((char) KeyEvent.VK_CLEAR);
				}
			}
		};
		KeyListener fileListener = new KeyListener() {
			@Override
			public void keyPressed(KeyEvent e) {}
			@Override
			public void keyReleased(KeyEvent e) {
				check_image(new File(path + File.separator + image.getText()));
			}
			@Override
			public void keyTyped(KeyEvent e) {}
		};
		// Textfields
		name = create_text("Name", new Rectangle(15, 5, 250, 15));
		image = create_text("Image", new Rectangle(15, 50, 225, 15));
		image.addKeyListener(fileListener);
		north = create_text("North*", new Rectangle(15, 100, 120, 15));
		north.addKeyListener(numListener);
		west = create_text("West*", new Rectangle(145, 100, 120, 15));
		west.addKeyListener(numListener);
		south = create_text("South*", new Rectangle(15, 140, 120, 15));
		south.addKeyListener(numListener);
		east = create_text("East*", new Rectangle(145, 140, 120, 15));
		east.addKeyListener(numListener);
		floor = create_text("Floor", new Rectangle(15, 200, 120, 15));
		floor.setText("0");
		floor.addKeyListener(numListener);
		priority = create_text("Priority", new Rectangle(15, 240, 120, 15));
		priority.setText("0");
		priority.addKeyListener(numListener);
		JLabel req_worldfile = new JLabel();
		req_worldfile.setBounds(18, 270, 150, 30);
		req_worldfile.setText("*required for world file");
		req_worldfile.setFont(new Font("Arial", Font.PLAIN, 11));
		// MapType
		JLabel lbl_maptype = new JLabel();
		lbl_maptype.setText("Map Type");
		lbl_maptype.setBounds(145, 200, 120, 15);
		lbl_maptype.setFont(new Font("Arial", Font.PLAIN, 11));
		maptype = new JComboBox();
		final String[] type = { "Buildings", "Landscape", "Others" };
		for (int i = 0; i < type.length; i++) {
			maptype.addItem(type[i]);
		}
		maptype.setFont(new Font("Arial", Font.PLAIN, 11));
		maptype.setBounds(145, 215, 120, 19);
		maptype.setSelectedIndex(0);
		
		final JComponent[] elements = { button_openkml, button_savekml, button_openwf, button_exit, icon,
				name, image, north, west, south, east, floor, priority,	req_worldfile, lbl_maptype, maptype};
		for (int i = 0; i < elements.length; i++) {
			panel.add(elements[i]);
		}
		add(panel);
		addWindowListener(new WindowListener() {
			@Override
			public void windowActivated(WindowEvent e) {}
			@Override
			public void windowClosed(WindowEvent e) {}
			@Override
			public void windowClosing(WindowEvent e) {
				save_and_exit();
			}
			@Override
			public void windowDeactivated(WindowEvent e) {}
			@Override
			public void windowDeiconified(WindowEvent e) {}
			@Override
			public void windowIconified(WindowEvent e) {}
			@Override
			public void windowOpened(WindowEvent e) {}
		});
		setVisible(true);
	}
	
	/**
	 * Import File.
	 */
	public void import_file() {
		JFileChooser fc = new JFileChooser() {
			private static final long serialVersionUID = 1L;
			@Override
			public void approveSelection() {
				File file = getSelectedFile();
				if (file.exists() && getDialogType() == SAVE_DIALOG) {
					int result = JOptionPane.showConfirmDialog(
							getTopLevelAncestor(),
							"The selected file already exists. Do you want to overwrite it?",
							"The file already exists",
							JOptionPane.YES_NO_CANCEL_OPTION,
							JOptionPane.QUESTION_MESSAGE);
					switch (result) {
					case JOptionPane.YES_OPTION:
						super.approveSelection();
						return;
					case JOptionPane.NO_OPTION:
						return;
					case JOptionPane.CANCEL_OPTION:
						cancelSelection();
						return;
					}
				}
				super.approveSelection();
			}
		};
		fc.setDialogTitle("WorldFileTool - Import...");
		fc.setFileFilter(new FileNameExtensionFilter("KML (*.kml) / Image (*.jpg; *.png; *.gif; *.tiff)",
														new String[] { "kml", "jpg", "jpeg", "png", "gif", "tif", "tiff" }));
		fc.setFileSelectionMode(JFileChooser.FILES_AND_DIRECTORIES);
		if (properties.containsKey(lastpath)) {
			fc.setCurrentDirectory(new File(properties.getProperty(lastpath)));
		}
		int result = fc.showOpenDialog(null);
		if (result == JFileChooser.APPROVE_OPTION) {
			file = fc.getSelectedFile();
			path = fc.getCurrentDirectory();
			properties.setProperty(lastpath, path.toString());
			if (file.getName().toLowerCase().endsWith("kml")) {
				clear();
				wft.read_kml(path.toString(), file);
				update_textfields();
				check_image(new File(path + File.separator + image.getText()));
			} else if (wft.check_ext(file, new String[] { "jpg", "jpeg", "png", "gif", "tif", "tiff" })) {
				clear();
				wft.read_image(path.toString(), file);
				update_textfields();
				check_image(new File(path + File.separator + image.getText()));
			}
		}
	}
	
	/**
	 * Export KML.
	 */
	public void export_kml() {
		if (!floor.getText().contains(".") && !floor.getText().isEmpty()) {
			if (99 < Integer.valueOf(floor.getText())) {
				error_message("WorldFileTool - Not valid Floor", "Choose an integer between -99 and 99 (default: 0)");
				return;
			}
		} else {
			error_message("WorldFileTool - Not valid Floor", "Choose an integer between -99 and 99 (default: 0)");
			return;
		} 
		if (!priority.getText().contains(".") && !priority.getText().isEmpty() && !priority.getText().contains("-")) {
			if (100 < Integer.valueOf(priority.getText())) {
				error_message("WorldFileTool - Not valid Priority", "Choose an integer between 1 and 100 or 0 for unknown");
				return;
			}
		} else {
			error_message("WorldFileTool - Not valid Priority", "Choose an integer between 1 and 100 or 0 for unknown");
			return;
		}
		if (name.getText().isEmpty() || image.getText().isEmpty() || north.getText().isEmpty()
				|| west.getText().isEmpty() || south.getText().isEmpty() || east.getText().isEmpty()) {
			error_message("WorldFileTool - Empty Textboxes", "Not all Textboxes are filled");
			return;
		}
		if (floor.getText().isEmpty())
			floor.setText("0");
		if (priority.getText().isEmpty())
			priority.setText("0");
		getContent();
		JFileChooser fc = new JFileChooser() {
			private static final long serialVersionUID = 1L;
			@Override
			public void approveSelection() {
				File file = getSelectedFile();
				if (file.exists() && getDialogType() == SAVE_DIALOG) {
					int result = JOptionPane.showConfirmDialog(
							getTopLevelAncestor(),
							"The selected file already exists. Do you want to overwrite it?",
							"The file already exists",
							JOptionPane.YES_NO_CANCEL_OPTION,
							JOptionPane.QUESTION_MESSAGE);
					switch (result) {
					case JOptionPane.YES_OPTION:
						super.approveSelection();
						return;
					case JOptionPane.NO_OPTION:
						return;
					case JOptionPane.CANCEL_OPTION:
						cancelSelection();
						return;
					}
				}
				super.approveSelection();
			}
		};
		fc.setFileFilter(new FileFilter() {
			@Override
			public boolean accept(File f) {
				return f.getName().toLowerCase().endsWith("kml");
			}
			@Override
			public String getDescription() {
				return "Google Earth KML-File (*.kml)";
			}
		});
		fc.setFileSelectionMode(JFileChooser.FILES_AND_DIRECTORIES);
		fc.setDialogTitle("Save as KML");
		if (properties.containsKey(lastpath)) {
			fc.setCurrentDirectory(new File(properties.getProperty(lastpath)));
		}
		final String[] ext = { "jpg", "jpeg", "png", "gif", "tif", "tiff" };
		if (file != null) {
			for (int i = 0; i < ext.length; i++) {
				if (file.getName().toLowerCase().endsWith(ext[i])) {
					fc.setSelectedFile(new File(Pattern.compile(ext[i], Pattern.CASE_INSENSITIVE).matcher(file.toString()).replaceAll("kml")));
					break;
				} else {
					fc.setSelectedFile(new File(file.toString()));
				}
			}
		}
		int result = fc.showSaveDialog(null);
		if (result == JFileChooser.APPROVE_OPTION) {
			file = fc.getSelectedFile();
			wft.write_kml(file);
		}
	}
	
	/**
	 * Export WorldFile.
	 */
	public void export_worldfile() {
		if (north.getText().isEmpty() || west.getText().isEmpty() || south.getText().isEmpty() || east.getText().isEmpty()) {
			error_message("WorldFileTool - Empty Textboxes", "Not all required Textboxes for a WorldFile are filled");
			return;
		}
		getContent();
		final String file = wft.rename(wft.image, "." + wft.ext, "." + wft.wf_ext);
		if (path != null) {
			if (!new File(file).exists()) {
				wft.write_wf(new File(path + File.separator + file));
			} else if (wft.ext.isEmpty() && wft.wf_ext.isEmpty()) {
				wft.write_wf(new File(path + File.separator + image.getText()));
			} else {
				int result = JOptionPane.showConfirmDialog(null, "The file already exists. Do you want to overwrite it?",
						"WorldFileTool - File already exists", JOptionPane.YES_NO_OPTION);
				switch(result) {
				case JOptionPane.YES_OPTION:
					wft.write_wf(new File(path + File.separator + file));
				case JOptionPane.NO_OPTION:
					return;
				}
			}
		} else {
			error_message("WorldFileTool - File not saved", "Could not save " + file);
		}
	}
	
	/**
	 * Create JButton.
	 * 
	 * @param text
	 * @param rect the rectangle
	 * @param event
	 * 
	 * @return the jbutton
	 */
	private JButton create_button(String text, Rectangle rect, int event) {
		JButton button = new JButton();
		button.setText(text);
		button.setBounds(rect);
		button.setToolTipText(text);
		button.setFont(new Font("Arial", Font.PLAIN, 11));
		button.setMnemonic(event);
		return button;
	}
	
	/**
	 * create textfields.
	 * 
	 * @param lbl the label
	 * @param rect the rectangle
	 * 
	 * @return JTextField
	 */
	private JTextField create_text(String lbl, Rectangle rect) {
		JLabel label = new JLabel();
		label.setText(lbl);
		label.setBounds(rect);
		label.setFont(new Font("Arial", Font.PLAIN, 11));
		panel.add(label);
		JTextField text = new JTextField();
		text.setBounds(rect.x, rect.y+15, rect.width, rect.height+5);
		JPopupMenu popupMenu = popupMenu(text);
		text.setComponentPopupMenu(popupMenu);
		return text;
	}
	
	/**
	 * Clear Textfields.
	 */
	private void clear() {
		name.setText("(unnamed)");
		image.setText("");
		north.setText("");
		west.setText("");
		south.setText("");
		east.setText("");
	}
	
	/**
	 * Update Textfields.
	 */
	private void update_textfields() {
		name.setText(wft.name);
		image.setText(wft.image);
		north.setText(wft.north);
		west.setText(wft.west);
		south.setText(wft.south);
		east.setText(wft.east);
	}
	
	/**
	 * prepare to write.
	 */
	private void getContent() {
		wft.name = name.getText().trim();
		wft.image = image.getText().trim();
		wft.north = north.getText().trim();
		wft.west = west.getText().trim();
		wft.south = south.getText().trim();
		wft.east = east.getText().trim();
		wft.floor = floor.getText().trim();
		wft.maptype = maptype.getSelectedItem().toString();
		if (wft.maptype.equalsIgnoreCase("Buildings")) {
			wft.maptype = "0";
		} else if (wft.maptype.equalsIgnoreCase("Landscape")) {
			wft.maptype = "1";
		} else if (wft.maptype.equalsIgnoreCase("Others")){
			wft.maptype = "2";
		}
		wft.priority = priority.getText().trim();
	}
	
	/**
	 * check if image file exists and display either a tick or cross.
	 * 
	 * @param file
	 */
	private void check_image(File file) {
		String pic;
		if (file.exists()) {
			pic = "/ch/hsr/worldfiletool/images/tick.png";
		} else {
			pic = "/ch/hsr/worldfiletool/images/cross.png";
		}
		icon.setIcon(new ImageIcon(getClass().getResource(pic)));
	}
	
	/**
	 * Error Message.
	 * 
	 * @param text
	 * @param message
	 */
	void error_message(String text, String message) {
		JOptionPane.showMessageDialog(null, message, text, JOptionPane.INFORMATION_MESSAGE);
	}
	
	/**
	 * setting cursor
	 */
	public void setCursor(int cursor) {
		setCursor(new Cursor(cursor));
	}
	
	/**
	 * save properties.
	 */
	private void save() {
		try {
			properties.store(new FileWriter("WorldFileTool.cfg"), "WorldFileTool Config");
		} catch (IOException e) {
			System.out.println(e);
		}
	}
	
	/**
	 * load properties.
	 */
	private void load() {
		try {
			properties.load(new FileReader("WorldFileTool.cfg"));
		} catch (FileNotFoundException e) {
			System.out.println(e);
		} catch (IOException e) {
			System.out.println(e);
		}
	}
	
	/**
	 * save properties and exit.
	 */
	private void save_and_exit() {
		save();
		String type = "";
		if (maptype.getSelectedItem().equals("Buildings")) {
			type = "0";
		} else if (maptype.getSelectedItem().equals("Landscape")) {
			type = "1";
		} else if (maptype.getSelectedItem().equals("Others")){
			type = "2";
		}
		String[] values = { name.getText(), image.getText(),
				north.getText(), west.getText(), south.getText(), east.getText(),
				floor.getText(), type, priority.getText() };
		int result = 0;
		if (!Arrays.asList(wft.values).containsAll(Arrays.asList(values))) {
			result = JOptionPane.showConfirmDialog(null, "Data has changed. Do you really want to quit?",
					"WorldFileTool - Quit?", JOptionPane.YES_NO_OPTION);
		}
		switch(result) {
		case JOptionPane.YES_OPTION:
			System.exit(0);
		case JOptionPane.NO_OPTION:
			return;
		}
	}
	
	/**
	 * MenuBar.
	 * 
	 * @return the JMenuBar
	 */
	private JMenuBar menu_bar() {
		// menu bar
		JMenuBar menuBar = new JMenuBar();
		
		// file menu
		JMenu fileMenu = new JMenu("File");
		fileMenu.setFont(new Font("Arial", Font.PLAIN, 11));
		// file menu items
		
		final String[] text = { "Import File...", "Export as KML...", "Export as World File" };
		final int[] event = { KeyEvent.VK_I, KeyEvent.VK_K, KeyEvent.VK_W };
		final String lbl_import = "import";
		final String lbl_kml = "kml";
		final String lbl_wf = "wf";
		String[] action = { lbl_import, lbl_kml, lbl_wf };
		
		ActionListener menuListener = new ActionListener() {
			@Override
			public void actionPerformed(ActionEvent e) {
				if (e.getActionCommand() == lbl_import) {
					import_file();
				} else if (e.getActionCommand() == lbl_kml) {
					export_kml();
				} else if (e.getActionCommand() == lbl_wf) {
					export_worldfile();
				}
			}
		};
		for (int i = 0; i < text.length; i++) {
			if (i == 1) {
				fileMenu.addSeparator();
			}
			JMenuItem menuItem = new JMenuItem();
			menuItem.setText(text[i]);
			menuItem.setAccelerator(KeyStroke.getKeyStroke(event[i], InputEvent.CTRL_DOWN_MASK));
			menuItem.setFont(new Font("Arial", Font.PLAIN, 11));
			menuItem.setActionCommand(action[i]);
			menuItem.addActionListener(menuListener);
			fileMenu.add(menuItem);
		}
		fileMenu.addSeparator();
		JMenuItem exitItem = new JMenuItem();
		exitItem.setText("Exit");
		exitItem.setFont(new Font("Arial", Font.PLAIN, 11));
		exitItem.addActionListener(new ActionListener() {
			@Override
			public void actionPerformed(ActionEvent e) {
				save_and_exit();
			}
		});
		
		fileMenu.add(exitItem);
		
		// help menu
		JMenu helpMenu = new JMenu("Help");
		helpMenu.setFont(new Font("Arial", Font.PLAIN, 11));
		// help menu items
		JMenuItem aboutItem = new JMenuItem();
		aboutItem.setText("About");
		aboutItem.setAccelerator(KeyStroke.getKeyStroke("F1"));
		aboutItem.setFont(new Font("Arial", Font.PLAIN, 11));
		aboutItem.addActionListener(new ActionListener() {
			@Override
			public void actionPerformed(ActionEvent e) {
				new About();
			}
		});
		helpMenu.add(aboutItem);
		menuBar.add(fileMenu);
		menuBar.add(helpMenu);
		
		return menuBar;
	}
	
	/**
	 * PopupMenu.
	 * 
	 * @param TextField
	 * 
	 * @return JPopupMenu
	 */
	private JPopupMenu popupMenu(final JTextField textField) {
		
		// PopupMenu on rightclick
		JPopupMenu popupMenu = new JPopupMenu();
		
		// PopupMenu Items
		final String[] menuItems = new String[] {"Cut", "Copy", "Paste", "Clear"};
		
		ActionListener mouseListener = new ActionListener() {
			@Override
			public void actionPerformed(ActionEvent event) {
				if (event.getActionCommand().equals("Cut")) {
					textField.cut();
				} else if (event.getActionCommand().equals("Copy")) {
					textField.copy();
				} else if (event.getActionCommand().equals("Paste")) {
					textField.paste();
				} else if (event.getActionCommand().equals("Clear")) {
					textField.setText("");
				}
			}
		};
		for (int i = 0; i < menuItems.length; i++) {
			JMenuItem menuItem = new JMenuItem(menuItems[i]);
			if (i == 3) {
				popupMenu.addSeparator();
			}
			menuItem.addActionListener(mouseListener);
			popupMenu.add(menuItem);
		}
		return popupMenu;
	}
}
