/**
 * 
 * World File Tool - A metadata management tool for georeferenced images.
 * Copyright (C) 2008
 * HSR University of Applied Science Rapperswil
 * IFS Institute for Software
 *
 * WorldFileTool is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * WorldFileTool is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with WorldFileTool.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package ch.hsr.worldfiletool;

import java.awt.Cursor;
import java.awt.image.BufferedImage;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.util.regex.Pattern;

import javax.imageio.ImageIO;

public class WorldFileTool {
	
	private static final String version = "0.3";
	String name = "";
	String image = "";
	String view = "0.75";
	String north = "";
	String west = "";
	String south = "";
	String east = "";
	String floor = "0";
	String maptype = "0";
	String priority = "0";
	private String input;
	double rot_y = 0.0;
	double rot_x = 0.0;
	double image_height = 0.0;
	double image_width = 0.0;
	String ext = "";
	String wf_ext = "";
	String[] values = { name, image, north, west, south, east, floor, maptype, priority };
	static GUI_Swing wftswing;
	private static final String wait = "wait";
	private static final String arrow = "arrow";
	
	/**
	 * The main method.
	 * 
	 * @param args the arguments
	 */
	public static void main(String[] args) {
		wftswing = new GUI_Swing();
		wftswing.create_gui(version);
	}
	
	/**
	 * Read KML.
	 * 
	 * @param path
	 * @param file
	 */
	public void read_kml(String path, File file) {
		clear();
		String line;
		boolean foundImage = false;
		StringBuffer temp = new StringBuffer();
		try {
			BufferedReader br = new BufferedReader(new FileReader(file));
			while((line = br.readLine()) != null) {
				temp.append(line + "\n");
			}
			br.close();
		} catch (FileNotFoundException e) {
			error_message("File Not Found", e.getMessage());
		} catch (IOException e) {
			System.out.println("read kml " + e);
		}
		input = temp.toString();
		name = extract("<name>", "</name>").trim();
		view = extract("<viewBoundScale>", "</viewBoundScale>").trim();
		image = extract("<href>", "</href>").trim();
		north = extract("<north>", "</north>").trim();
		west = extract("<west>", "</west>").trim();
		south = extract("<south>", "</south>").trim();
		east = extract("<east>", "</east>").trim();
		String rotation = extract("<rotation>", "</rotation>").trim();
		final String[] ext = { "jpg", "jpeg", "png", "gif", "tif", "tiff" };
		if(check_ext(new File(image), ext)) {
			get_image_data(new File(path + File.separator + image));
			foundImage = true;
		}
		if(!foundImage) {
			error_message("WorldFileTool - " + image + " not found",
						"The image should be either in JPG, PNG, GIF or TIFF and in a relative path.");
		} else if(foundImage && !rotation.isEmpty() && !rotation.equals("0")) {
			try {
				if (!north.isEmpty() && !south.isEmpty() && !west.isEmpty() && !east.isEmpty()) {
					File imagefile = new File(path + File.separator + image);
					File imagefile_ori = new File(rename(path + File.separator + image, "." + this.ext, "_ori." + this.ext));
					imagefile.renameTo(imagefile_ori);
					ImageIO.write(ImageIO.read(imagefile_ori), this.ext, imagefile);
					setCursor(wait);
					ImageIO.write(RotateImage.rotate_image(new File(path + File.separator + image), Double.valueOf(rotation)), this.ext, new File(path + File.separator + image));
					setCursor(arrow);
					calc_new_values(path);
				} else {
					error_message("Verify your coordinates",
							"Could not read coordinates from kml therefore no rotation has been made");
				}
			} catch (IOException e) {
				error_message("WorldFileTool - " + image + " not found",
									"The image should be either in JPG, PNG, GIF or TIFF and in a relative path.");
				return;
			}
		}
		save_data();
	}

	/**
	 * Read Image.
	 * 
	 * @param path
	 * @param file
	 */
	public void read_image(String path, File file) {
		clear();
		get_image_data(file);
		if (check_ext(file, new String[] { "jpg", "jpeg" })) {
			read_worldfile(rename(file.toString(), "jpe?g", "jgw"));
		} else if (check_ext(file, new String[] { "png" })) {
			read_worldfile(rename(file.toString(), "png", "pgw"));
		} else if (check_ext(file, new String[] { "gif" })) {
			read_worldfile(rename(file.toString(), "gif", "gfw"));
		} else if (check_ext(file, new String[] { "tif", "tiff" })) {
			read_worldfile(rename(file.toString(), "tiff?", "tfw"));
		}
		if (!(rot_x == 0 && rot_y == 0)) {
			try {
				File imagefile = new File(image);
				File imagefile_ori = new File(rename(image, "." + this.ext, "_ori." + this.ext));
				imagefile.renameTo(imagefile_ori);
				ImageIO.write(ImageIO.read(imagefile_ori), this.ext, imagefile);
				setCursor(wait);
				ImageIO.write(RotateImage.rotate_vectors(file, rot_x, rot_y), this.ext, new File(image));
				setCursor(arrow);
				calc_new_values(path);
			} catch (IOException e) {
				error_message("WorldFileTool - " + image + " not found",
									"The image should be either in JPG, PNG, GIF or TIFF and in a relative path.");
				return;
			}
		}
		save_data();
	}
	
	/**
	 * calc new values after rotation,
	 * because the image size changed
	 * 
	 * @param path
	 */
	private void calc_new_values(String path) {
		double north_d = Double.valueOf(north);
		double south_d = Double.valueOf(south);
		double west_d = Double.valueOf(west);
		double east_d = Double.valueOf(east);
		// save image size
		double image_height = this.image_height;
		double image_width = this.image_width;
		// image height in deegree
		double diff_vert = Math.abs(north_d - south_d);
		double diff_horiz = Math.abs(west_d - east_d);
		// deegree per pixel
		double yFactor = diff_vert / image_height;
		double xFactor = diff_horiz / image_width;
		// get image size after rotation
		get_image_data(new File(path + File.separator + image));
		// calc height/ width difference 
		double diff_height = Math.abs(image_height - this.image_height);
		double diff_width = Math.abs(image_width - this.image_width);
		// add the difference to the corresponding direction
		if (north_d > south_d) {
			north = String.valueOf(north_d + diff_height*yFactor/2);
			south = String.valueOf(south_d - diff_height*yFactor/2);
		} else if (south_d > north_d){
			north = String.valueOf(north_d - diff_height*yFactor/2);
			south = String.valueOf(south_d + diff_height*yFactor/2);
		}
		if (west_d > east_d) {
			west = String.valueOf(west_d + diff_width*xFactor/2);
			east = String.valueOf(east_d - diff_width*xFactor/2);
		} else if (east_d > west_d) {
			west = String.valueOf(west_d - diff_width*xFactor/2);
			east = String.valueOf(east_d + diff_width*xFactor/2);
		}
	}
	
	/**
	 * Rename file.
	 * 
	 * @param file
	 * @param regex
	 * @param repl the replacement
	 */
	String rename( String file, String regex, String repl) {
		return Pattern.compile(regex, Pattern.CASE_INSENSITIVE).matcher(file).replaceAll(repl);
	}
	
	/**
	 * check file extension.
	 * 
	 * @param file
	 * @param ext the extension
	 * 
	 * @return true, if successful
	 */
	boolean check_ext(File file, String[] ext) {
		boolean found = false;
		for (int i = 0; i < ext.length; i++) {
			if (file.getName().toLowerCase().endsWith(ext[i])) {
				if (ext[i].equalsIgnoreCase("jpg")
						|| ext[i].equalsIgnoreCase("jpeg")) {
					wf_ext = "jgw";
				} else if (ext[i].equalsIgnoreCase("png")) {
					wf_ext = "pgw";
				} else if (ext[i].equalsIgnoreCase("gif")) {
					wf_ext = "gfw";
				} else if (ext[i].equalsIgnoreCase("tif")
						|| ext[i].equalsIgnoreCase("tiff")) {
					wf_ext = "tfw";
				}
				found = true;
				this.ext = ext[i];
				break;
			}
		}
		return found;
	}
	
	/**
	 * Read WorldFile.
	 * 
	 * @param file
	 */
	private void read_worldfile(String file) {
		double pix_x = 0.0;
		double pix_y = 0.0;
		double x_coord = 0.0;
		double y_coord = 0.0;
		try {
			BufferedReader br = new BufferedReader(new FileReader(file));
			pix_x = Double.valueOf(br.readLine().trim());
			rot_y = Double.valueOf(br.readLine().trim());
			rot_x = Double.valueOf(br.readLine().trim());
			pix_y = Double.valueOf(br.readLine().trim());
			x_coord = Double.valueOf(br.readLine().trim());
			y_coord = Double.valueOf(br.readLine().trim());
			br.close();
		} catch (FileNotFoundException e) {
			error_message("WorldFileTool - File not found", e.getMessage());
			return;
		} catch (IOException e) {
			error_message("WorldFileTool - Could not write file", e.getMessage());
			return;
		}
		if (!(-180 < x_coord && x_coord < 180) || !(-90 < y_coord && y_coord < 90)) {
			error_message("WorldFileTool - Wrong coordinate system", "Convert your coordinates to the geographic coordinate system and retry");
		} else {
			north = String.valueOf(y_coord);
			west = String.valueOf(x_coord);
			east = String.valueOf(x_coord + image_width*pix_x);
			south = String.valueOf(y_coord + image_height*pix_y);
		}
	}
	
	/**
	 * extract file input.
	 * 
	 * @param start start tag
	 * @param end end tag
	 * 
	 * @return string
	 */
	private String extract(String start, String end) {
		String str = "";
		if (input.contains(start)) {
			String[] split = input.split(start);
			split = split[1].split(end);
			str = split[0];
		}
		return str;
	}
	
	/**
	 * read image data.
	 * 
	 * @param file
	 */
	private void get_image_data(File file) {
		image = file.getName().toString();
		try {
			BufferedImage bi = ImageIO.read(file);
			image_height = bi.getHeight();
			image_width = bi.getWidth();
		} catch (IOException e) {
			System.out.println("get_image_data " + e);
		}
	}
	
	/**
	 * Write KML.
	 * 
	 * @param file
	 */
	public void write_kml(File file) {
		input = 
			"<?xml version=\"1.0\" encoding=\"utf-8\"?>" + "\n" +
			"<kml xmlns=\"http://earth.google.com/kml/2.2\">" + "\n" +
			"<GroundOverlay>" + "\n" +
			"    <name>" + name + "</name>" + "\n" +
			"    <Icon>" + "\n" +
			"        <href>" + image + "</href>" + "\n" +
			"        <viewBoundScale>" + view + "</viewBoundScale>" + "\n" +
			"    </Icon>" + "\n" +
			"    <LatLonBox>" + "\n" +
			"        <north>" + north + "</north>" + "\n" +
			"        <west>" + west + "</west>" + "\n" +
			"        <south>" + south + "</south>" + "\n" +
			"        <east>" + east + "</east>" + "\n" +
			"    </LatLonBox>" + "\n" +
			"    <Metadata>" + "\n" +
			"        <floor>" + floor + "</floor>" + "\n" +
			"        <background>" + maptype + "</background>" + "\n" +
			"        <priority>" + priority + "</priority>" + "\n" +
			"    </Metadata>" + "\n" +
			"</GroundOverlay>" + "\n" +
			"</kml>" + "\n";
		try {
			BufferedWriter bw = new BufferedWriter(new FileWriter(file));
			bw.write(input);
			bw.close();
		} catch (IOException e) {
			error_message("WorldFileTool - File not saved", e.getMessage());
		}
	}
	
	/**
	 * write worldfile.
	 * 
	 * @param file
	 */
	public void write_wf(File file) {
		double y_coord = Double.valueOf(north);
		double x_coord = Double.valueOf(west);
		double pix_x = (Double.valueOf(east) - x_coord)/ image_width;
		double pix_y = (Double.valueOf(south) - y_coord)/ image_height;
		input =
			pix_x + "\n" +
			0 + "\n" +
			0 + "\n" +
			pix_y + "\n" +
			x_coord + "\n" +
			y_coord + "\n";
		try {
			BufferedWriter bw = new BufferedWriter(new FileWriter(file));
			bw.write(input);
			bw.close();
		} catch (IOException e) {
			error_message("WorldFileTool - File not saved", e.getMessage());
			return;
		}
		error_message("WorldFileTool - File saved", "Successfully exported to " + file);
	}
	
	/**
	 * save parsed input data for close mechanism.
	 */
	private void save_data() {
		values = new String[] { name, image, north, west, south, east, floor, maptype, priority };
	}

	/**
	 * Clear.
	 */
	private void clear() {
		name = "(unnamed)";
		image = "";
		north = "";
		west = "";
		south = "";
		east = "";
	}
	
	/**
	 * error message in swing.
	 * 
	 * @param text
	 * @param message
	 */
	private void error_message(String text, String message) {
		wftswing.error_message(text, message);
	}
	
	/**
	 * set cursor in swing.
	 * 
	 * @param cursor the new cursor
	 */
	private void setCursor(String cursor) {
		if (cursor == wait) {
			wftswing.setCursor(Cursor.WAIT_CURSOR);
		} else if (cursor == arrow) {
			wftswing.setCursor(Cursor.DEFAULT_CURSOR);
		}
	}
}
