# -*- coding: utf-8 -*-

import pyproj
import numpy as np
import osgeo.gdal as gdal


class Proj:
    """
    Eine Klasse die die Projektion von Koordinaten etwas vereinfacht.
    Zuerst eine Instanz der Klasse erstellen mit den korrekten 
    Koordinatensystemen (EPSG-Code als String angeben), und dann die Koordinaten
    projezieren:
    wgs84_to_ch1903 = Proj(4326, 21781)
    wgs84_to_ch1903.transform(6.1, 46.5)
    """
    def __init__(self, from_srs, to_srs):
        self.s_srs = pyproj.Proj(init='EPSG:'+str(from_srs))
        self.t_srs = pyproj.Proj(init='EPSG:'+str(to_srs))
    
    def transform(self, x, y):
        return pyproj.transform(self.s_srs, self.t_srs, x, y)




def coord_to_pixel(coord, env, res):
    """
    Transformiert die Geo-Koordinaten in Pixel-Koordinaten.
    Dazu brauchen wir nicht nur die Koordinaten, sondern auch die Envelope
    (Tupel mit xmin, ymin, xmax, ymax), und die Raster-Auflösung (res).
    """
    x = int((coord[0] - env[0]) / res)
    y = int((coord[1] - env[1]) / res)
    return x, y




def write_raster(outfile, nparray, env, res):
    """
    Schreibt einen Numpy-Array in eine Erdas-Imagine-Datei.
    Die GeoTransform enthält eine Liste
    [upper_left_x, resolution_x, rotation_1, upper_left_y, rotation_2, -resolution_y]
    Die Rotations-Parameter sind in einem geraden Bild 0.
    """
    driver = gdal.GetDriverByName('HFA')
    nx, ny = nparray.shape
    dataset = driver.Create(outfile, xsize=nx, ysize=ny, 
        bands=1, eType = gdal.GDT_Float32)
    dataset.SetGeoTransform([env[0], res, 0.0, env[3], 0.0, -res])
    band1 = dataset.GetRasterBand(1)
    band1.WriteArray(np.flipud(nparray.T))

