# -*- coding: utf-8 -*-
"""
mietpreis.py
Berechnet den durchschnittlichen Mietpreis pro m2 auf einem Raster
das über die ganze Schweiz geht. Ausgangspunkt ist die Textdatei mit
den Wohnungsinseraten (appartements.txt).
Dabei sind die Koordinaten der Wohnungen in Lat/Long, und wir konvertieren
sie hier in Schweizer Koordinaten.
"""

# Input-Parameter:
infile = 'appartements.txt'
outfile = 'appartements.img'    # Raster-Datei im Erdas Imagine Format
env = 480000, 75000, 860000, 300000     # Envelope: xmin, ymin, xmax, ymax
res = 1000  # Resolution: Grösse einer Rasterzelle


import numpy as np
import geo          # Dies ist unser eigenes Modul (geo.py)


# Raster-Grösse berechnen (wie viele Zellen in x und y?)
nx = (env[2] - env[0]) / res
ny = (env[3] - env[1]) / res

# Um den durchschnittlichen Mietpreis pro m2 zu berechnen brauchen wir folgende
# Informationen:
# surface_habitable (m2), loyer_mois (CHF)
# Wir müssen für jede Rasterzelle die Summe von CHF/m2 speichern, die Anzahl
# Wohnungen, und den Durchschnitt (Summe / Anzahl). Also brauchen wir drei
# Numpy-Arrays.
summe = np.zeros((nx,ny))
anzahl = np.zeros((nx,ny))
durchschnitt = np.zeros((nx,ny))

# Die Projektion vorbereiten.
wgs84_to_ch1903 = geo.Proj(from_srs=4326, to_srs=21781)

# Die Input-Datei öffnen
fp = open(infile)
h = fp.readline()       # Den Header mit den Variablennamen lesen

# Zeile für Zeile der Input-Datei behandeln
for zeile in fp:
    werte = zeile.split('\t')
    # Falls wir weniger als 5 Werte haben, Zeile überspringen:
    if len(werte) < 5: continue
    # Falls eine Kolonne einen fehlenden Wert hat Zeile überspringen:
    if werte[4] == 'NULL' or werte[5] == 'NULL': continue
    lat, lng = float(werte[1]), float(werte[2])
    flaeche = float(werte[4])
    mietpreis = float(werte[5])
    # Falls der Mietpreis über 20'000 Franken ist, haben wir es wahrscheinlich
    # mit einer Falschinformation zu tun. Also Zeile überspringen
    if mietpreis > 20000: continue
    # Nun können wir die Koordinaten in Schweizer Koordinaten umwandeln:
    x, y = wgs84_to_ch1903.transform(lng, lat)
    # Und die dazugehörigen Pixel-Koordinaten finden:
    px, py = geo.coord_to_pixel([x,y], env, res)
    # Den Mietpreis pro m2 berechnen:
    mietpreis_m2 = mietpreis / flaeche
    # Und nun die drei Numpy-Arrays updaten:
    summe[px, py] += mietpreis_m2
    anzahl[px, py] += 1
    durchschnitt[px, py] = summe[px, py] / anzahl[px, py]


# Wir sind durch die ganze Datei durch. Also Datei schliessen.
fp.close()

# Und nun den Numpy-Array in eine Erdas-Imagine Datei schreiben:
geo.write_raster(outfile, durchschnitt, env, res)
    

